/** @file   scatterevent.cpp
 * @brief   Implementation of ScatterEvent class
 * @version $Revision: 1.1.1.1 $
 * @date    $Date: 2006/01/21 23:02:46 $
 * @author  Tomi Lamminsaari
 */


#include "scatterevent.h" // class's header file
#include "eng2d.h"
#include "gameobject.h"
#include "warglobals.h"
#include "www_map.h"
#include "door.h"
#include <list>
#include "www_assert.h"
using std::string;
using std::istream;
using std::vector;
using eng2d::Vec2D;
using eng2d::Rect2D;
using std::list;


namespace WeWantWar {


/** Constructor.
 */
ScatterEvent::ScatterEvent() :
  DynamicEvent()
{
}



/** Destructor.
 */
ScatterEvent::~ScatterEvent()
{
  for (int i=0; i < m_scatterList.size(); i++) {
    delete m_scatterList.at(i);
  }
  m_scatterList.clear();
}



/** Reads the scatter data
 */
int ScatterEvent::readData( istream& fin )
{
  LOG_MESSAGE( "Reading Scatter-event" );
  while ( true ) {
    if ( fin.eof() ) {
      return -1;
    }
    
    string tmp;
    fin >> tmp;
    if ( tmp == "</scatter>" ) {
      return 0;
      
    } else if ( tmp == "#" ) {
      fin.ignore( 4096, '\n' );
      
    } else if ( tmp == "<gameobjects>" ) {
      ScatterItem* pSI = this->readScatterGameObjects( fin );
      if ( pSI == 0 ) {
        return -1;
      }
      m_scatterList.push_back( pSI );
      
    } else if ( tmp == "<killed>" ) {
      ScatterItem* pSI = this->readScatterKilled( fin );
      if ( pSI == 0 ) {
        return -1;
      }
      m_scatterList.push_back( pSI );
      
    } else if ( tmp == "<id>" ) {
      ScatterItem* pSI = this->readScatterID( fin );
      if ( pSI == 0 ) {
        return -1;
      }
      m_scatterList.push_back( pSI );
      
    } else if ( tmp == "<door>" ) {
      ScatterItem* pSI = this->readScatterDoor( fin );
      if ( pSI == 0 ) {
        return -1;
      }
      m_scatterList.push_back( pSI );
      
    } else if ( tmp == "<bonus>" ) {
      ScatterItem* pSI = this->readScatterBonus( fin );
      if ( pSI == 0 ) {
        return -1;
      }
      m_scatterList.push_back( pSI );
      
    } else {
      int ret = alert("Unsupported Scatter-item", tmp.c_str(), 0,
                      "Continue", "Quit", 0,0 );
      if ( ret == 2 ) {
        return -1;
      }
    }
  }
}



/** Updates this event
 */
void ScatterEvent::update()
{
  if ( this->status() == DynamicEvent::ACTIVE ) {
    DynamicEvent::m_counter -= 1;
    if ( DynamicEvent::m_counter < 0 ) {
      this->scatter();
      this->status( DynamicEvent::COMPLETE );
    }
  }
}



/** Activates this event
 */
void ScatterEvent::activate()
{
  if ( this->status() == DynamicEvent::COMPLETE ) {
    return;
  }
  this->status( DynamicEvent::ACTIVE );
}



/** DOes the scatter-action.
 */
void ScatterEvent::scatter()
{
  for ( int i=0; i < m_scatterList.size(); i++ ) {
    ScatterItem* pSI = m_scatterList.at( i );
    switch ( pSI->mode ) {
      case ( SCATTER_GAMEOBJECTS ): {
        this->scatterGameObjects( pSI );
        break;
      }
      case ( SCATTER_KILLED ): {
        this->scatterKilled( pSI );
        break;
      }
      case ( SCATTER_ID ): {
        this->scatterID( pSI );
        break;
      }
      case ( SCATTER_DOOR ): {
        this->scatterDoors( pSI );
        break;
      }
      case ( SCATTER_BONUS ): {
        this->scatterBonus( pSI );
        break;
      }
    }
  }
}



/** Scatters the gameobjects inside the rectangle.
 */
void ScatterEvent::scatterGameObjects( ScatterItem* pSI )
{
  // First we collect the objects we should remove to a list.
  list<GameObject*> remObjects;
  for ( int i=0; i < WarGlobals::pObjTable->objectList.size(); i++ ) {
    GameObject* pO = WarGlobals::pObjTable->objectList.at( i );
    if ( pSI->rect.pointInside( pO->position() ) == true ) {
      remObjects.push_back( pO );
    }
  }
  // And then we remove the objects from the global object table.
  list<GameObject*>::const_iterator it = remObjects.begin();
  while ( it != remObjects.end() ) {
    WarGlobals::pObjTable->remObject( (*it) );
    it++;
  }
}



/** Scatters the killed gameobjects inside the rectangle.
 */
void ScatterEvent::scatterKilled( ScatterItem* pSI )
{
  // First we collect the objects we should remove to a list.
  list<GameObject*> remObjects;
  for ( int i=0; i < WarGlobals::pObjTable->objectList.size(); i++ ) {
    GameObject* pO = WarGlobals::pObjTable->objectList.at( i );
    if ( pSI->rect.pointInside( pO->position() ) == true &&
         pO->state() == GameObject::STATE_KILLED ) {
      remObjects.push_back( pO );
    }
  }
  // And then we remove the objects from the global object table.
  list<GameObject*>::const_iterator it = remObjects.begin();
  while ( it != remObjects.end() ) {
    WarGlobals::pObjTable->remObject( (*it) );
    it++;
  }
}



/** Scatters the gameobjects inside the rectangle.
 */
void ScatterEvent::scatterID( ScatterItem* pSI )
{
  // First we collect the objects we should remove to a list.
  list<GameObject*> remObjects;
  for ( int i=0; i < WarGlobals::pObjTable->objectList.size(); i++ ) {
    GameObject* pO = WarGlobals::pObjTable->objectList.at( i );
    if ( pSI->rect.pointInside( pO->position() ) == true &&
         pO->objectID() == pSI->idcode ) {
      remObjects.push_back( pO );
    }
  }
  // And then we remove the objects from the global object table.
  list<GameObject*>::const_iterator it = remObjects.begin();
  while ( it != remObjects.end() ) {
    WarGlobals::pObjTable->remObject( (*it) );
    it++;
  }
}



/** Scatters the doors
 */
void ScatterEvent::scatterDoors( ScatterItem* pSI )
{
  // Collect the removable doors in a list
  list<Door*> remDoors;
  for ( int i=0; i < WarGlobals::doorList.size(); i++ ) {
    Door* pD = WarGlobals::doorList.at( i );
    Rect2D doorRect( pD->getDoorArea() );
    if ( doorRect.intersects( pSI->rect ) == true ) {
      remDoors.push_back( pD );
    }
  }
  // And delete them from the door-table.
  list<Door*>::const_iterator it = remDoors.begin();
  while ( it != remDoors.end() ) {
    vector<Door*>::iterator it2 = WarGlobals::doorList.begin();
    while ( it2 != WarGlobals::doorList.end() ) {
      if ( (*it) == (*it2) ) {
        WarGlobals::doorList.erase( it2 );
        break;
      }
      it2++;
    }
    it++;
  }
}



/** Scatters the bonuses
 */
void ScatterEvent::scatterBonus( ScatterItem* pSI )
{
  // Collect the removable bonuses in a list
  list<BonusObject*> remBonus;
  for ( int i=0; i < WarGlobals::bonusList.size(); i++ ) {
    BonusObject* pB = WarGlobals::bonusList.at( i );
    if ( pSI->rect.pointInside( pB->position() ) == true ) {
      remBonus.push_back( pB );
    }
  }
  // And delete them from the bonus-table.
  list<BonusObject*>::const_iterator it = remBonus.begin();
  while ( it != remBonus.end() ) {
    vector<BonusObject*>::iterator it2 = WarGlobals::bonusList.begin();
    while ( it2 != WarGlobals::bonusList.end() ) {
      if ( (*it) == (*it2) ) {
        WarGlobals::bonusList.erase( it2 );
        break;
      }
      it2++;
    }
    it++;
  }
}


/** Reads the data of SCATTER_GAMEOBJECTS
 */
ScatterEvent::ScatterItem* ScatterEvent::readScatterGameObjects( istream& rIn )
{
  ScatterItem* pSI = new ScatterItem;
  pSI->mode = SCATTER_GAMEOBJECTS;
  pSI->idcode = 0;
  pSI->rect.topleft = Vec2D( 0,0 );
  pSI->rect.bottomright = Vec2D( Map::getWidth(Map::IN_PIXELS),
                                 Map::getHeight(Map::IN_PIXELS) );
  pSI->rect.bottomright *= 32;
  
  while ( true ) {
    if ( rIn.eof() == true ) {
      alert( "EOF encountered while parsing", "Scatter:GAMEOBJECTS - element", 0,
             "Quit", 0, 0,0);
      delete pSI;
      return 0;
    }
    
    string tmp;
    rIn >> tmp;
    if ( tmp == "</gameobjects>" ) {
      return pSI;
      
    } else if ( tmp == "#" ) {
      rIn.ignore( 4096, '\n' );
      
    } else if ( tmp == "rect:" ) {
      rIn >> pSI->rect.topleft.vx >> pSI->rect.topleft.vy;
      rIn >> pSI->rect.bottomright.vx >> pSI->rect.bottomright.vy;
      pSI->rect.topleft *= 32;
      pSI->rect.bottomright *= 32;
      
    } else {
      int ret = alert( "Unsupported parameter for Scatter:GAMEOBJECTS",
                       tmp.c_str(), 0, "Continue", "Quit", 0,0);
      if ( ret == 2 ) {
        delete pSI;
        return 0;
      }
    }
  }
}



/** Reads the data of SCATTER_KILLED element
 */
ScatterEvent::ScatterItem* ScatterEvent::readScatterKilled( istream& rIn )
{
  ScatterItem* pSI = new ScatterItem;
  pSI->mode = SCATTER_KILLED;
  pSI->idcode = 0;
  pSI->rect.topleft = Vec2D( 0,0 );
  pSI->rect.bottomright = Vec2D( Map::getWidth(Map::IN_PIXELS),
                                 Map::getHeight(Map::IN_PIXELS) );
  pSI->rect.bottomright *= 32;

  while ( true ) {
    if ( rIn.eof() == true ) {
      alert( "EOF encountered while parsing", "Scatter:KILLED - element", 0,
             "Quit", 0, 0,0);
      delete pSI;
      return 0;
    }

    string tmp;
    rIn >> tmp;
    if ( tmp == "</killed>" ) {
      return pSI;

    } else if ( tmp == "#" ) {
      rIn.ignore( 4096, '\n' );

    } else if ( tmp == "rect:" ) {
      rIn >> pSI->rect.topleft.vx >> pSI->rect.topleft.vy;
      rIn >> pSI->rect.bottomright.vx >> pSI->rect.bottomright.vy;
      pSI->rect.topleft *= 32;
      pSI->rect.bottomright *= 32;

    } else {
      int ret = alert( "Unsupported parameter for Scatter:KILLED",
                       tmp.c_str(), 0, "Continue", "Quit", 0,0);
      if ( ret == 2 ) {
        delete pSI;
        return 0;
      }
    }
  }
}



/** Reads the data of SCATTER_ID event
 */
ScatterEvent::ScatterItem* ScatterEvent::readScatterID( istream& rIn )
{
  ScatterItem* pSI = new ScatterItem;
  pSI->mode = SCATTER_ID;
  pSI->idcode = -1;
  pSI->rect.topleft = Vec2D( 0,0 );
  pSI->rect.bottomright = Vec2D( Map::getWidth(Map::IN_PIXELS),
                                 Map::getHeight(Map::IN_PIXELS) );
  pSI->rect.bottomright *= 32;

  while ( true ) {
    if ( rIn.eof() == true ) {
      alert( "EOF encountered while parsing", "Scatter:KILLED - element", 0,
             "Quit", 0, 0,0);
      delete pSI;
      return 0;
    }

    string tmp;
    rIn >> tmp;
    if ( tmp == "</killed>" ) {
      return pSI;

    } else if ( tmp == "#" ) {
      rIn.ignore( 4096, '\n' );

    } else if ( tmp == "rect:" ) {
      rIn >> pSI->rect.topleft.vx >> pSI->rect.topleft.vy;
      rIn >> pSI->rect.bottomright.vx >> pSI->rect.bottomright.vy;
      pSI->rect.topleft *= 32;
      pSI->rect.bottomright *= 32;
      
    } else if ( tmp == "id:" ) {
      rIn >> pSI->idcode;

    } else {
      int ret = alert( "Unsupported parameter for Scatter:KILLED",
                       tmp.c_str(), 0, "Continue", "Quit", 0,0);
      if ( ret == 2 ) {
        delete pSI;
        return 0;
      }
    }
  }
}



/** Reads the data of SCATTER_DOOR event
 */
ScatterEvent::ScatterItem* ScatterEvent::readScatterDoor( istream& rIn )
{
  ScatterItem* pSI = new ScatterItem;
  pSI->mode = SCATTER_DOOR;
  pSI->idcode = 0;
  pSI->rect.topleft = Vec2D( 0,0 );
  pSI->rect.bottomright = Vec2D( Map::getWidth(Map::IN_PIXELS),
                                 Map::getHeight(Map::IN_PIXELS) );
  pSI->rect.bottomright *= 32;

  while ( true ) {
    if ( rIn.eof() == true ) {
      alert( "EOF encountered while parsing", "Scatter:KILLED - element", 0,
             "Quit", 0, 0,0);
      delete pSI;
      return 0;
    }

    string tmp;
    rIn >> tmp;
    if ( tmp == "</killed>" ) {
      return pSI;

    } else if ( tmp == "#" ) {
      rIn.ignore( 4096, '\n' );

    } else if ( tmp == "rect:" ) {
      rIn >> pSI->rect.topleft.vx >> pSI->rect.topleft.vy;
      rIn >> pSI->rect.bottomright.vx >> pSI->rect.bottomright.vy;
      pSI->rect.topleft *= 32;
      pSI->rect.bottomright *= 32;

    } else {
      int ret = alert( "Unsupported parameter for Scatter:KILLED",
                       tmp.c_str(), 0, "Continue", "Quit", 0,0);
      if ( ret == 2 ) {
        delete pSI;
        return 0;
      }
    }
  }
}



/** Reads the data of SCATTER_BONUS event
 */
ScatterEvent::ScatterItem* ScatterEvent::readScatterBonus( istream& rIn )
{
}


} // end of namespace
